package easik.ui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Image;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.ImageProducer;
import java.net.URL;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import easik.Easik;
import easik.sketch.Sketch;
import easik.sketch.constraint.Constraint;
import easik.sketch.constraint.ProductConstraint;
import easik.sketch.constraint.SumConstraint;
import easik.sketch.edge.SketchEdge;
import easik.sketch.vertex.EntityNode;
import easik.states.BasicEditingState;
import easik.ui.menu.AboutAction;
import easik.ui.menu.AddCommutativeDiagramAction;
import easik.ui.menu.AddProductConstraintAction;
import easik.ui.menu.AddPullbackConstraintAction;
import easik.ui.menu.AddSumConstraintAction;
import easik.ui.menu.DocumentInfoAction;
import easik.ui.menu.EditDataTypesAction;
import easik.ui.menu.ExportSQLAction;
import easik.ui.menu.ExportSQLTextAction;
import easik.ui.menu.ExportTexAction;
import easik.ui.menu.FileNewAction;
import easik.ui.menu.FileOpenAction;
import easik.ui.menu.FileQuitAction;
import easik.ui.menu.FileSaveAction;
import easik.ui.menu.FileSaveAsAction;
import easik.ui.menu.HelpAction;
import easik.ui.menu.HideAllConstraintsAction;
import easik.ui.menu.RecentFileAction;
import easik.ui.menu.ShowAllConstraintsAction;
import easik.ui.menu.popup.DeleteAction;
import easik.ui.menu.popup.NewEdgeAction;
import easik.ui.menu.popup.NewEntityAction;
import easik.ui.menu.popup.NewInjectiveEdgeAction;
import easik.ui.menu.popup.RenameAction;
import easik.ui.tree.InfoTreeUI;
import easik.ui.tree.popup.AddAttributeAction;
import easik.ui.tree.popup.AddPathAction;
import easik.ui.tree.popup.AddUniqueKeyAction;
import easik.ui.tree.popup.HideConstraint;

/** 
 * This is the swing starting point for the program. This frame contains the 
 * entire application. It is accessible through the JEase singleton by all other
 * classes.
 * 
 * @author Rob Fletcher 2005
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @version 2006-08-04 Kevin Green
 */
public class ApplicationFrame extends JFrame {
	/**
	 * The popup menu found on the graphical sketch
	 */
	private JPopupMenu _popupMenu;
	/**
	 * The point where the popup menu should appear
	 */
	private Point _popupPosition;
	/**
	 * The current sketch
	 */
	private Sketch _ourSketch;
	/**
	 * The right panel of the frame
	 */
	private JPanel _rightPane;
	/**
	 * The button panel of the frame
	 */
	private JPanel _buttonPane;
	/**
	 * The main panel of the frame
	 */
	private JPanel _mainPane;
	/**
	 * The scroll pane of the sketch
	 */
	private JScrollPane _sketchScrollPane;
	/**
	 * The information tree
	 */
	private InfoTreeUI _infoTreeUI;
	/**
	 * The status bar
	 */
	private JLabel _statusBar;
	/**
	 * The state label
	 */
	private JLabel _stateLabel;
	/**
	 * The tree name
	 */
	private JLabel _treeName;
	/**
	 * The next button
	 */
	private JButton _nextButton;
	/**
	 * The finish button
	 */
	private JButton _finishButton;
	/**
	 * The cancel button
	 */
	private JButton _cancelButton;
	/**
	 * The split pane
	 */
	private JSplitPane _mainSplitPane;
	/**
	 * The recent files menu created from the recent files stored in the ini file
	 */
	private JMenu _recentFilesMenu;
	/**
	 * The show constraints menu item
	 */
	private JMenuItem _showConstraintsMenu;
	/**
	 * The hide constraints menu item
	 */
	private JMenuItem _hideConstraintsMenu;
	/**
	 * The MySQL menu item
	 */
	private JCheckBoxMenuItem _MySQLMenu;
	/**
	 * The Oracle menu item
	 */
	private JCheckBoxMenuItem _OracleMenu; 
	/**
	 * The DB2 menu item
	 */
	private JCheckBoxMenuItem _DB2Menu;
	/**
	 * The XML menu item
	 */
	private JCheckBoxMenuItem _XMLMenu;
	/**
	 * The User defined menu item
	 */
	private JCheckBoxMenuItem _UserDefinedMenu;
	/**
	 * The Show Attribute & Unique Keys menu item
	 */
	private JCheckBoxMenuItem _ShowAttsMenu;
	/**
	 * The Add Entity popup menu item
	 */
	private JMenuItem _AddEntityPopItem;
	/**
	 * The Add Entity popup menu item
	 */
	private JMenuItem _AddEdgePopItem;
	/**
	 * The Add Edge popup menu item
	 */
	private JMenuItem _AddInjEdgePopItem;
	/**
	 * The Add Injective Edge popup menu item
	 */
	private JMenuItem _RenamePopItem;
	/**
	 * The Rename popup menu item
	 */
	private JMenuItem _DeletePopItem;
	/**
	 * The Delete popup menu item
	 */
	private JMenuItem _AddAttributePopItem;
	/**
	 * The Add Unique Key popup menu item
	 */
	private JMenuItem _AddUniqueKeyPopItem;
	/**
	 * The Hide Constraint popup menu item
	 */
	private JMenuItem _HideConstraintPopItem;
	/**
	 * The Next popup menu item
	 */
	private JMenuItem _NextPopItem;
	/**
	 * The Finish popup menu item
	 */
	private JMenuItem _FinishPopItem;
	/**
	 * The cancel popup menu item
	 */
	private JMenuItem _CancelPopItem;
	/**
	 * The add paths popup menu item
	 */
	private JMenuItem _AddPathPopItem;
	/**
	 * The Program Settings menu item
	 */
	private JMenuItem _ProgramSettingsItem;
	
	
	/**
	 *	Creates the application frame, sets some sizes, sets up the properties
	 *	builds the menus, and lays out the swing components.
	 */
	public ApplicationFrame() {
		super("Easik - Untitled");
		
		try{
			URL url = Easik.class.getResource( "easik.jpg" );
			Image myImg = Toolkit.getDefaultToolkit().createImage( ( ImageProducer ) url.getContent() );
			this.setIconImage(myImg);
		}
		catch(Exception e){
			System.out.println("Error loading image");
		}
		
		this.setSize(700, 600);
		this.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
		
		this.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent evt) {
                closeWindow();
            }
        });

		//Initialize all variables
		_ourSketch = new Sketch(this);
		_popupMenu = new JPopupMenu();
		_popupPosition = new Point(0, 0);
		_nextButton = new JButton("Next");
		_cancelButton = new JButton("Cancel");
		_finishButton = new JButton("Finish");
		_rightPane = new JPanel();
		_buttonPane = new JPanel();
		_mainPane = new JPanel();
		_infoTreeUI = new InfoTreeUI();
		
		_infoTreeUI.refreshTree();
		
		//Build Menus
		buildMenu();
		buildPopupMenu();
		
		//Setup Sketch Pane
		_sketchScrollPane = new JScrollPane(_ourSketch);
		_sketchScrollPane.setMinimumSize(new Dimension(300, 300));
		
		//Setup Status Bar
		_statusBar = new JLabel("Status bar");
		
		//Setup button pane
		_buttonPane.setLayout(new GridLayout(1,0));
		_stateLabel = new JLabel("Default Text");
		_buttonPane.add(new JLabel("   "));
		_buttonPane.add(_stateLabel);
		_buttonPane.add(_nextButton);
		_buttonPane.add(_finishButton);
		_buttonPane.add(_cancelButton);
		_buttonPane.setVisible(false);
		
		//Button action listener setup
		_nextButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				Easik.getInstance().getStateManager().nextClicked();
			}
		});

		_finishButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e1){
				Easik.getInstance().getStateManager().finishClicked();
			}
		});
		_cancelButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				Easik.getInstance().getStateManager().cancelClicked();
			}
		});

		//Setup entire right pane
		_rightPane.setLayout(new BorderLayout());
		_treeName = new JLabel("   " + _ourSketch.getDocInfo().getName());
		_rightPane.add(_treeName, BorderLayout.NORTH);
		_rightPane.add(_infoTreeUI, BorderLayout.CENTER);
		
		//Setup main pane
		_mainPane.setLayout(new BorderLayout());
		_mainPane.add(_sketchScrollPane, BorderLayout.CENTER);
		_mainPane.add(_buttonPane, BorderLayout.SOUTH);
		
		//Setup enitre window
		_mainSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, _mainPane, _rightPane);
		_mainSplitPane.setDividerLocation(445);
		_mainSplitPane.setDividerSize(10);
		_mainSplitPane.setResizeWeight(1);
		_mainSplitPane.setOneTouchExpandable(true);
		_mainSplitPane.setContinuousLayout(true);
		getContentPane().add(_mainSplitPane, BorderLayout.CENTER);
		getContentPane().add(_statusBar, BorderLayout.SOUTH);
		
	}
	
	/**
	 * Trys to close window as long it is not dirty.
	 */
	public void closeWindow(){
		int n = 0;
		if (Easik.getInstance().getFrame().getSketch().getDirty()) {
			n =
				JOptionPane.showConfirmDialog(
					Easik.getInstance().getFrame(),
					"Unsaved changes will be lost, continue anyway?",
					"Caution!",
					JOptionPane.YES_NO_OPTION);			
		}
		if (n == 0) {			
			Easik.getInstance().getIni().writeToFile();
			System.exit(0);
		}
	}
	
	/**
	 * Sets if the button pane is visible or not
	 * 
	 * @param isVis True if visible, false if hidden.
	 */
	public void setButtonPaneVisibility(boolean isVis){
		_buttonPane.setVisible(isVis);
	}
	
	/**	 
	 *	The state label indicates the current state above the next/cancel buttons.
	 *	this method changes that text.
	 * 
	 * @param inState
	 */
	public void setStateString(String inState) {
		_stateLabel.setText(inState);
	}
	
	/**
	 * Sets the String to be displayed above the information tree.
	 * 
	 * @param name The name of the tree
	 * @since 2006-07-14 Vera Ranieri
	 */
	public void setTreeName(String name){
		_treeName.setText("   " + name);
	}

	/**
	 * Returns the cancel button.
	 * 
	 * @return The cancel button
	 */
	public JButton getCancelButton() {
		return _cancelButton;
	}
	
	/**
	 * Returns the next button
	 * 
	 * @return The next button
	 */
	public JButton getNextButton() {
		return _nextButton;
	}

	/**
	 * Returns the finish button
	 * 
	 * @return The finish button
	 * @since 2006-05-30 Vera Ranieri
	 */
	public JButton getFinishButton(){
		return _finishButton;
	}
	
	/**
	 * Updates what is displayed on the status bar. Takes a string
	 * 
	 * @param newStatus
	 */
	public void setStatusBar(String newStatus) {
		_statusBar.setText(newStatus);
	}

	/**
	 * Set the Show Constraints menu item to reflect if constraints are shown.
	 * 
	 * @param setting True if constraints should be shown, false otherwise.
	 */
	public void setShowConstraints(boolean setting){
		_showConstraintsMenu.setSelected(setting);
	}
	/**
	 * Returns the current sketch
	 * 
	 * @return The sketch
	 */
	public Sketch getSketch() {
		return _ourSketch;
	}
	
	/**
	 * Returns the InfoTreeUI object
	 * 
	 * @return The InfoTreeUI object
	 */
	public InfoTreeUI getInfoTreeUI(){
		return _infoTreeUI;
	}
	
	/**
	 * Sets the InfoTreeUI
	 * 
	 * @param inInfoTreeUI The new InfoTreeUI
	 */
	public void setInfoTreeUI(InfoTreeUI inInfoTreeUI){
		_rightPane.remove(_infoTreeUI);
		_infoTreeUI = inInfoTreeUI;
		_rightPane.add(_infoTreeUI);
		_mainSplitPane.setDividerLocation(_mainSplitPane.getDividerLocation());
	}
	
	/**
	 * Creates the popup menu
	 */
	private void buildPopupMenu() {
		_popupMenu.add(_AddEntityPopItem = new JMenuItem(new NewEntityAction(_popupPosition)));
		_popupMenu.add(_AddEdgePopItem = new JMenuItem(new NewEdgeAction()));
		_popupMenu.add(_AddInjEdgePopItem = new JMenuItem(new NewInjectiveEdgeAction()));
		_popupMenu.add(_AddAttributePopItem = new JMenuItem(new AddAttributeAction()));
		_popupMenu.add(_AddUniqueKeyPopItem= new JMenuItem(new AddUniqueKeyAction()));
		_popupMenu.add(_RenamePopItem = new JMenuItem(new RenameAction()));
		_popupMenu.add(_DeletePopItem = new JMenuItem(new DeleteAction()));
		_popupMenu.add(_HideConstraintPopItem = new JMenuItem(new HideConstraint()));
		_popupMenu.add(_NextPopItem = new JMenuItem("Next"));
		_popupMenu.add(_FinishPopItem = new JMenuItem("Finish"));
		_popupMenu.add(_CancelPopItem = new JMenuItem("Cancel"));
		_popupMenu.add(_AddPathPopItem = new JMenuItem(new AddPathAction()));
		
		_NextPopItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				Easik.getInstance().getStateManager().nextClicked();
			}
		});
		_FinishPopItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				Easik.getInstance().getStateManager().finishClicked();
			}
		});
		_CancelPopItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				Easik.getInstance().getStateManager().cancelClicked();
			}
		});

		_ourSketch.addMouseListener(new MouseAdapter() {

			public void mousePressed(MouseEvent evt) {
				if (evt.isPopupTrigger()) {
					_popupPosition.setLocation(evt.getX(), evt.getY());
					setPopMenuItems();
					_popupMenu.show(evt.getComponent(), evt.getX(), evt.getY());
				}
			}
			public void mouseReleased(MouseEvent evt) {
				if (evt.isPopupTrigger()) {
					_popupPosition.setLocation(evt.getX(), evt.getY());
					setPopMenuItems();
					_popupMenu.show(evt.getComponent(), evt.getX(), evt.getY());
				}
			}
		});
	}
	
	/**
	 * Sets the visibility of popup menu items based on what is selected
	 */
	public void setPopMenuItems(){
		Sketch _ourSketch = Easik.getInstance().getFrame().getSketch();
		
		//Hide all elements
		for(Component c : _popupMenu.getComponents()){
			c.setVisible(false);
		}
		
		if(Easik.getInstance().getFrame().getNextButton().isEnabled()){
			_NextPopItem.setVisible(true);
		}
		if(Easik.getInstance().getFrame().getFinishButton().isEnabled()){
			_FinishPopItem.setVisible(true);
		}
		if(Easik.getInstance().getFrame().getCancelButton().isEnabled()){
			_CancelPopItem.setVisible(true);
		}
		
		if(Easik.getInstance().getStateManager().peekState().getClass() == BasicEditingState.class){
			Vector currentSelection = _ourSketch.getSelectionListener().getCurrentSelection();
			if(currentSelection.size() > 0){
				_DeletePopItem.setVisible(true);
			}
			else{
				_AddEntityPopItem.setVisible(true);
			}
			
			if(currentSelection.size() == 1){
				if(currentSelection.get(0) instanceof EntityNode){
					EntityNode myEntity = (EntityNode)currentSelection.get(0);
					DefaultMutableTreeNode myNode = myEntity.getNode();
					if(myNode == null)
						return;
					TreeNode[] myNodes = myNode.getPath();
					if(myNodes == null)
						return;
					TreePath myPath = new TreePath(myNodes);
					Easik.getInstance().getFrame().getInfoTreeUI().getInfoTree().scrollPathToVisible(myPath);
					Easik.getInstance().getFrame().getInfoTreeUI().getInfoTree().setSelectionPath(myPath);
					_AddAttributePopItem.setVisible(true);
					_AddUniqueKeyPopItem.setVisible(true);
				}	
				if(currentSelection.get(0) instanceof Constraint){
					Constraint myConst = (Constraint)currentSelection.get(0);
					if(myConst instanceof SumConstraint || myConst instanceof ProductConstraint){
						_AddPathPopItem.setVisible(true);
					}
					DefaultMutableTreeNode myNode = myConst.getNode();
					if(myNode == null)
						return;
					TreeNode[] myNodes = myNode.getPath();
					if(myNodes == null)
						return;
					TreePath myPath = new TreePath(myNodes);
					Easik.getInstance().getFrame().getInfoTreeUI().getInfoTree().scrollPathToVisible(myPath);
					Easik.getInstance().getFrame().getInfoTreeUI().getInfoTree().setSelectionPath(myPath);
					_HideConstraintPopItem.setVisible(true);
				}
				if(currentSelection.get(0) instanceof EntityNode || currentSelection.get(0) instanceof SketchEdge){
					_RenamePopItem.setVisible(true);
				}		
			}
			else if(currentSelection.size() == 2){
				if(currentSelection.get(0) instanceof EntityNode && currentSelection.get(1) instanceof EntityNode){
					_AddEdgePopItem.setVisible(true);
					_AddInjEdgePopItem.setVisible(true);
				}
			}
		}
	}

	/**
	 * Builds the menu
	 */
	private void buildMenu() {
		JMenuBar mainMenu;
		JMenu menuFile, menuAction, menuHelp, menuDocument, menuSettings;

		mainMenu = new JMenuBar();

		//Make the File Menu
		menuFile = new JMenu("File");
		mainMenu.add(menuFile);
		menuFile.setMnemonic(KeyEvent.VK_F);
		menuFile.add(new JMenuItem(new FileNewAction()));
		menuFile.addSeparator();
		menuFile.add(new JMenuItem(new FileOpenAction()));
		menuFile.add(new JMenuItem(new FileSaveAction()));
		menuFile.add(new JMenuItem(new FileSaveAsAction()));
		menuFile.addSeparator();
		
		JMenu exportMenu = new JMenu("Export...");
		menuFile.add(exportMenu);
		exportMenu.add(new JMenuItem(new ExportSQLAction()));
		exportMenu.add(new JMenuItem(new ExportSQLTextAction()));
		exportMenu.add(new JMenuItem(new ExportTexAction()));
		menuFile.addSeparator();
		
		_recentFilesMenu = new JMenu("Recent Files...");
		menuFile.add(_recentFilesMenu);
		updateRecentFilesMenu();
		menuFile.addSeparator();
		
		menuFile.add(new JMenuItem(new FileQuitAction()));
		
		//Make the document menu
		menuDocument = new JMenu("Document");
		menuDocument.setMnemonic(KeyEvent.VK_D);
		mainMenu.add(menuDocument);
		menuDocument.add(new JMenuItem(new DocumentInfoAction()));
		menuDocument.addSeparator();
		menuDocument.add(new JMenuItem(new EditDataTypesAction()));
		menuDocument.addSeparator();
		
		//Create data type output sub menu
		JMenu dataTypeMenu = new JMenu("Enable Output Data Types");
		menuDocument.add(dataTypeMenu);
		_MySQLMenu = new JCheckBoxMenuItem("MySQL", _ourSketch.getDataTypeController().is_useMySQL());
		_MySQLMenu.setToolTipText("Toggle Use of MySQL Data Types");
		_OracleMenu = new JCheckBoxMenuItem("Oracle", _ourSketch.getDataTypeController().is_useOracle());
		_OracleMenu.setToolTipText("Toggle Use of Oracle Data Types");
		_DB2Menu = new JCheckBoxMenuItem("DB2", _ourSketch.getDataTypeController().is_useDB2());
		_DB2Menu.setToolTipText("Toggle Use of DB2 Data Types");
		_XMLMenu = new JCheckBoxMenuItem("XML", _ourSketch.getDataTypeController().is_useXML());
		_XMLMenu.setToolTipText("Toggle Use of XML Data Types");
		_UserDefinedMenu = new JCheckBoxMenuItem("User Defined", _ourSketch.getDataTypeController().is_useUserDefined());
		_UserDefinedMenu.setToolTipText("Toggle Use of User Defined Data Types");
		
		dataTypeMenu.add(_MySQLMenu);
		dataTypeMenu.add(_OracleMenu);
		dataTypeMenu.add(_DB2Menu);
		dataTypeMenu.add(_XMLMenu);
		dataTypeMenu.add(_UserDefinedMenu);
		
		//Add action listener for data type output sub menu
		_MySQLMenu.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				if(_ourSketch.getDataTypeController().is_useMySQL()){
					_ourSketch.getDataTypeController().set_useMySQL(false);
					_MySQLMenu.setSelected(false);
				}
				else{
					_ourSketch.getDataTypeController().set_useMySQL(true);
					_MySQLMenu.setSelected(true);
					_ourSketch.getDataTypeController().loadNullsFromDefaults("MySQL");
				}
			}
		});
		_OracleMenu.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				if(_ourSketch.getDataTypeController().is_useOracle()){
					_ourSketch.getDataTypeController().set_useOracle(false);
					_OracleMenu.setSelected(false);
				}
				else{
					_ourSketch.getDataTypeController().set_useOracle(true);
					_OracleMenu.setSelected(true);
					_ourSketch.getDataTypeController().loadNullsFromDefaults("Oracle");
				}
			}
		});
		_DB2Menu.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				if(_ourSketch.getDataTypeController().is_useDB2()){
					_ourSketch.getDataTypeController().set_useDB2(false);
					_DB2Menu.setSelected(false);
				}
				else{
					_ourSketch.getDataTypeController().set_useDB2(true);
					_DB2Menu.setSelected(true);
					_ourSketch.getDataTypeController().loadNullsFromDefaults("DB2");
				}
			}
		});
		_XMLMenu.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				if(_ourSketch.getDataTypeController().is_useXML()){
					_ourSketch.getDataTypeController().set_useXML(false);
					_XMLMenu.setSelected(false);
				}
				else{
					_ourSketch.getDataTypeController().set_useXML(true);
					_XMLMenu.setSelected(true);
					_ourSketch.getDataTypeController().loadNullsFromDefaults("XML");
				}
			}
		});
		_UserDefinedMenu.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				if(_ourSketch.getDataTypeController().is_useUserDefined()){
					_ourSketch.getDataTypeController().set_useUserDefined(false);
					_UserDefinedMenu.setSelected(false);
				}
				else{
					_ourSketch.getDataTypeController().set_useUserDefined(true);
					_UserDefinedMenu.setSelected(true);
					_ourSketch.getDataTypeController().loadNullsFromDefaults("UserDefined");
				}
			}
		});
		
		//Create constraints menu
		menuAction = new JMenu("Constraints");
		menuAction.setMnemonic(KeyEvent.VK_C);
		mainMenu.add(menuAction);
		
		menuAction.add(new JMenuItem(new AddCommutativeDiagramAction()));
		menuAction.add(new JMenuItem(new AddSumConstraintAction()));
		menuAction.add(new JMenuItem(new AddProductConstraintAction()));
		menuAction.add(new JMenuItem(new AddPullbackConstraintAction()));
		menuAction.addSeparator();
		//So we can dynamically change this if necessary.
		_showConstraintsMenu = new JMenuItem(new ShowAllConstraintsAction());
		menuAction.add(_showConstraintsMenu);
		_hideConstraintsMenu = new JMenuItem(new HideAllConstraintsAction());
		menuAction.add(_hideConstraintsMenu);

		//Creat Settings menu
		menuSettings = new JMenu("Settings");
		mainMenu.add(menuSettings);
		menuSettings.add(_ProgramSettingsItem = new JMenuItem("Program Settings"));
		_ProgramSettingsItem.setToolTipText("Set Global Easik Properties");
		_ProgramSettingsItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				ProgramSettingsUI myUI = new ProgramSettingsUI();
				myUI.setVisible(true);
			}
		});
		menuSettings.addSeparator();
		menuSettings.add(_ShowAttsMenu = new JCheckBoxMenuItem("Show all attributes & unique keys"));
		_ShowAttsMenu.setToolTipText("Toggle Graphical Display of Attributes and Unique Keys");
		_ShowAttsMenu.setState(Easik.getInstance().getIni().isShowAllAttsAndUniqueKeys());
		
		_ShowAttsMenu.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				Object[] myCell;
				myCell = _ourSketch.getGraphLayoutCache().getCells(_ourSketch.getGraphLayoutCache().getCellViews());
				_ourSketch.getGraphLayoutCache().hideCells(myCell, true);
				_ourSketch.getGraphLayoutCache().showCells(myCell, true);
				_ourSketch.repaint();
			}
		});
		
		//Create help menu
		menuHelp = new JMenu("Help");
		mainMenu.add(menuHelp);
		menuHelp.setMnemonic(KeyEvent.VK_H);
		menuHelp.add(new HelpAction(_ourSketch, this));
		menuHelp.add(new AboutAction());

		setJMenuBar(mainMenu);
	}
	
	/**
	 * Updates the recent files menu items
	 */
	public void updateRecentFilesMenu(){
		_recentFilesMenu.removeAll();
		for(String file : Easik.getInstance().getIni().getRECENT_FILES()){
			_recentFilesMenu.add(new JMenuItem(new RecentFileAction(file)));
		}
	}
	
	/**
	 * Refreshes the values of the check boxes based on the current sketch
	 */
	public void refreshPlatformCheckBoxes(){
		_MySQLMenu.setSelected(_ourSketch.getDataTypeController().is_useMySQL());
		_OracleMenu.setSelected(_ourSketch.getDataTypeController().is_useOracle());
		_DB2Menu.setSelected(_ourSketch.getDataTypeController().is_useDB2());
		_XMLMenu.setSelected(_ourSketch.getDataTypeController().is_useXML());
		_UserDefinedMenu.setSelected(_ourSketch.getDataTypeController().is_useUserDefined());
	}
	
	/**
	 * Returns the value of the show attributes check box menu item
	 * 
	 * @return The value of the show attributes check box menu item
	 */
	public boolean getShowAttsVal(){
		return _ShowAttsMenu.getState();
	}
}
